# $Id: Crypto.pm 33634 2011-01-24 16:41:10Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/squarepeg/lib/UvT/Squarepeg/Crypto.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Squarepeg::Crypto::Config;

use Xyzzy -self;

field secret => sub { die "no Secret configured\n" };
*set_secret = *secret;

package UvT::Squarepeg::Crypto;

use MIME::Base64;
use Digest::SHA qw(sha256);
use Crypt::OpenSSL::Random;
use Time::HiRes qw(time);

use Clarity -self;

field cfg;
field secret => sub { shift->cfg->secret };

sub random_bytes {
	die "Insufficient entropy\n"
		unless Crypt::OpenSSL::Random::random_status;
	return Crypt::OpenSSL::Random::random_bytes(@_);
}

sub random_hex {
	my $hex = unpack('H*', $self->random_bytes($@));
	$hex =~ tr/a-z/A-Z/;
	return $hex;
}

sub create_token {
	my ($hidden, @keys) = @_;

	foreach(@keys) {
		utf8::encode($_);
		die "Bad characters in request\n"
			unless index($_, $;) < 0;
	}

	my $time = pack('Q>', int(time * 1e6));
	my $salt = $self->random_bytes(16);
	my $token = $time . $salt . join($;, @keys);

	my $hash = sha256($self->secret, $;, $hidden, $;, $token);

	my $encoded = encode_base64($hash.$token, '');
	$encoded =~ s/=+$//;
	$encoded =~ tr|/+|._|;

	return $time, $salt, $encoded;
}

sub check_token {
	my ($hidden, $token, $expiry) = @_;

	die "Invalid token\n"
		unless defined $token;

	die "Invalid token\n"
		unless $token =~ /^[a-zA-Z0-9_.]+$/;

	$token =~ tr|._|/+|;
	$token = decode_base64($token.'====');

	my $hash = substr($token, 0, 32, '');
	die "Invalid token\n"
		unless length($hash) == 32;

	die "Invalid token\n"
		unless sha256($self->secret, $;, $hidden, $;, $token) eq $hash;

	my $time = unpack('Q>', substr($token, 0, 8, '')) / 1e6;

	my $now = time;

	die "Invalid token\n"
		if $time > $now;

	die "Expired token\n"
		if $expiry && $time + $expiry < $now;

	my $salt = substr($token, 0, 16, '');

	return $time, $salt, map { utf8::decode($_); $_ } split($;, $token)
}
