# $Id: Directory.pm 37965 2012-10-10 11:44:00Z anton $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/pwdmodifier/pwdmodifier/lib/UvT/PwdModifier/Directory.pm $
use strict;
use warnings FATAL => 'all';

package UvT::PwdModifier::Directory::Entry;
use Data::Dumper;
use MIME::Base64;
use Digest::SHA1;
use Spiffy -Base;

field 'ldap';
field 'cf';

sub orgstats {
	my %res;

	my $cf = $self->cf;
	my @config = $cf->value('ldap_orgstat');
	die $cf->error if $cf->error;

	my $attr = shift @config;

	foreach my $status ($self->ldap->get_value($attr)) {
		foreach my $line (@config) {
			my ($name, $regex) = split(' ', $line, 2);
			undef $res{$name}
				if $status =~ /$regex/i;
		}
	}
	return (keys %res);
}

sub laus {
	return $self->ldap->get_value('uvt-lau');
}

sub uid {
	return scalar $self->ldap->get_value('uid');
}

sub mail {
	return scalar $self->ldap->get_value('mail');
}

sub anr {
	return scalar $self->ldap->get_value('employeeNumber');
}

sub accountStatus {
	return $self->ldap->get_value('accountStatus');
}

sub accountSuspended {
	return $self->ldap->get_value('accountSuspended');
}

sub tiasPrivateEmail {
	return $self->ldap->get_value('tiasPrivateEmail');
}

sub accountTemporaryDisabled {
	return $self->ldap->get_value('accountTemporaryDisabled');
}

sub stuff {
	my $entry = $self->ldap;

	my %userinfo;
	foreach my $attr (
		qw(
		accountSuspended
		accountTemporaryDisabled
		cn
		datePasswdChanged
		mail
		mailhost
		privateEmail
		sn
		tiasPrivateEmail
		uid )) {
		$userinfo{$attr} = $entry->get_value($attr);
	}

    $userinfo{dn} = $entry->dn;

    $userinfo{orgstatus} = [$entry->get_value('organizationalStatus;lang-nl')];
    $userinfo{orgstats} = $self->orgstats;

    $userinfo{ou} = [$entry->get_value('ou;lang-nl')];
	$userinfo{'uvt-lau'} = [$entry->get_value('uvt-lau')];
	$userinfo{entry} = $entry;

    # NB er zijn 'niet studenten' met een studentenaccount op de studentenserver!
    # kijk dus NIET naar de organizationalStatus, want dan vallen deze buiten de boot.
    my $uid = $userinfo{uid};
    $userinfo{studentaccount} = $uid =~ /^(s\d{6}|u\d{7})$/i;

	return \%userinfo;
}

package UvT::PwdModifier::Directory;

use Spiffy -Base;

use Net::LDAP;
use Net::LDAP::Filter;

field 'cf', -init => 'die "no cf configured\n"';
field 'ldap';

sub ldapError {
	# spiffy stopt de eerste parameter in $self
	my $res = $self;
	die $res->error unless $res->code == 4;
	return $res;
}


sub ldap_connection {
	my $cf = $self->cf;
	my $ldap = $self->ldap;
	unless($ldap) {
		my $server = $cf->value('ldaphost');
		my $capath = $cf->value('capath');
		my $retries = $cf->value('ldapRetries');
		die $cf->error if $cf->error;

		my $attempt = 0;
		my $state;

		warn "Connecting to ldapserver: $server";
		$ldap = new Net::LDAP($server,
#								  onerror => \&ldapError,
								  timeout => 10,
								  verify => 'require',
								  capath => $capath );
		$state = $@;
		die "Error connecting to $server: $state" unless $ldap;

		$ldap->start_tls(verify => 'require', capath => $capath)
			unless $ldap->cipher;

		$self->ldap($ldap);
	}
	return $ldap;
}

sub ldap_search_filter {
	my ($filter) = @_;
	my $ldap = $self->ldap_connection();
	# NB deze search vindt ALTIJD plaats vanuit een anonymous bind
	$ldap->bind;

	my $cf = $self->cf;
	my $base = $cf->value('ldapbase');
	die $cf->error if $cf->error;

	my $search = $ldap->search(base => $base, filter => $filter);
	my @entries = $search->entries;

	return undef, "no match found for filter: '$filter'"
		unless @entries;

	return undef, "multiple entries found for filter: '$filter'"
		if @entries > 1;

	return $entries[0], undef;
}

sub ldap_search_uid {
	my ($uid) = @_;

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'uid', assertionValue => $uid}},
		'Net::LDAP::Filter')->as_string;

	return $self->ldap_search_filter($filter);
}

sub ldap_search_anr {
	my ($anr) = @_;

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'employeeNumber', assertionValue => $anr}},
		'Net::LDAP::Filter')->as_string;

	return $self->ldap_search_filter($filter);
}



sub verify_password {
	my ($victim, $password, $trySHA) = @_;
	my $ldap = $self->ldap_connection();
	$ldap->bind;

	my ($entry, $err) = $self->ldap_search_uid($victim);
	return $err unless ($entry);

	my $dn = $entry->dn;
	eval { $ldap->bind($dn, password => $password) };
	$err = $@;
	if ($err and $trySHA) {
		my $ctx = Digest::SHA1->new;
		$ctx->add($password.$trySHA);
		my $digest = $ctx->digest;
		my $converted = MIME::Base64::encode_base64($digest);
		# vergeet niet de \n eraf te slopen!
		chomp $converted;
		eval { $ldap->bind($dn, password => $converted) };
		$err = $@;
	}
	$ldap->bind;
	return $err;
}


sub get_entry {
	my $uid = shift;
	my ($entry, $err) = $self->ldap_search_uid($uid);

	return undef, $err
		unless $entry;

	return new UvT::PwdModifier::Directory::Entry(ldap => $entry, cf => $self->cf), undef;
}

sub check_uvt_auth {
	my ($admin, $auths) = @_;

	my $ldap = $self->ldap_connection();
	$ldap->bind;

	# (&(uid=bob)(|(uvt-auth=pwd_change/foo)(uvt-auth=pwd_change/foo/bar)))
	my $filter = bless(
			{and => [{equalityMatch => {attributeDesc => 'uid', assertionValue => $admin}},
				{or => [map { {equalityMatch => {attributeDesc => 'uvt-auth', assertionValue => $_}} } @$auths]}
		]}, 'Net::LDAP::Filter')->as_string;

	# 3) zoek op dat filter
	my ($entry, $err) = $self->ldap_search_filter($filter);
	return $entry ? undef : $err;
}


sub bindAccount {
	my ($secretconfig, $ldap) = @_;
	my $bindAccount = $secretconfig->value('bindAccount');
	my ($filter, $pwd) = split (/\s*;\s*/, $bindAccount,2);
	my ($entry, $err) = $self->ldap_search_filter($filter);

	my $dn = $entry->dn;
	my $msg = $ldap->bind($dn, password => $pwd);
	if ($msg->code) {
		die "bind failed:", $msg->error;
	}
	warn "Bind success for $dn";
}


sub getLdapEntries {
	my ($secretconfig, $filter) = @_;
	my $cf = $self->cf;
	my $base = $cf->value('ldapbase');
	my $ldap = $self->ldap_connection();
	$self->bindAccount($secretconfig, $ldap);
	my $res = $ldap->search(base => $base, filter => $filter);
	return $res;
}

sub getUIDs {
	my $msg = $self->getLdapEntries(@_);
	return [map {$_->get_value('uid')} $msg->entries];
}

sub listTemporaryDisabled {
	my $secretconfig = shift;
	my $filter = '(AccountTemporaryDisabled=*)';
	return $self->getUIDs($secretconfig, $filter);
}

sub listSuspended {
	my $secretconfig = shift;
	# verander * eventueel in een anr om te testen
#	my $filter = '(&(accountSuspended=*)(userpassword=*)(employeeNumber=*))';
	my $filter = '(&(accountSuspended=*)(!(rcryptpassword=suspended*))(employeeNumber=*))';

	return $self->getUIDs($secretconfig, $filter);
}

sub listUnsuspended {
	my $secretconfig = shift;
#	my $filter = '(&(objectclass=uvtUser)(employeeNumber=*)(!(accountSuspended=*))(!(userpassword=*)))';
	my $filter = '(&(employeeNumber=*)(!(accountSuspended=*))(|(!(userpassword=*))(rcryptpassword=suspended*)))';
	return $self->getUIDs($secretconfig, $filter);
}

sub getLdapAccountByUid {
	my $uid = shift;
	my $ldap = $self->ldap_connection();
	my $cf = $self->cf;
	my $base = $cf->value('ldapbase');

	my $filter = bless(
			{equalityMatch => {attributeDesc => 'uid', assertionValue => $uid}},
		'Net::LDAP::Filter')->as_string;

	my $search = $ldap->search(base => $base, filter => $filter);
	my @entries = $search->entries;

	die "no match found for filter: '$filter'"
		unless @entries;

	die "multiple entries found for filter: '$filter'"
		if @entries > 1;

	return $entries[0];
}


#sub markSuspendedStateInLdap {
#	my ($params, $unsuspend) = @_;
#	my $cf = $self->cf;
#	my $ldap = $self->ldap_connection();
#	$self->bindAccount($cf, $ldap);
#	my $uid = $params->{userinfo}->{uid};
##	warn Dumper($params);
#	my $entry = $self->getLdapAccountByUid($uid);
#	my $attrib = 'uvt-auth';
#	my $value = 'accountActivelySuspended';
#	my @values = $entry->get_value($attrib);
#	my $dn = $entry->dn;
#	my $msg;
#	if ($unsuspend){
#		$entry->delete({"$attrib","$value"}) if grep($value, @values);
#	} else {
#		$msg = $ldap->modify($dn, add => {"$attrib","$value"}) unless grep($value, @values);
#	}
#
##	warn "still alive";
##	my $msg = $entry->update($ldap);
#	if ($msg->code) {
#		die("markSuspendeStateInLdap failed:", $msg->error());
#	} else {
#		warn "markSuspendeStateInLdap succeeded";
#	}
#}

sub changeAccountTemporaryDisabledState {
	my ($secretconfig, $uid, $admin, $on ) = @_;
	my $attrib = 'accountTemporaryDisabled';
	my $ldap = $self->ldap_connection();
	$self->bindAccount($secretconfig, $ldap);

	my $cf = $self->cf;
	my $base = $cf->value('ldapbase');
	die $cf->error if $cf->error;

	my $entry = $self->getLdapAccountByUid($uid);

	if ($on) {
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
		$year += 1900; $mday +=1 ;
		my $content = sprintf("Disabled at: $year-%02d-%02d %02d:%02d by: $admin", $mon, $mday, $hour,$min);
		$entry->replace($attrib, $content);
	} else {
		$entry->delete($attrib);
	}

	my $msg = $entry->update($ldap);
	if ($msg->code) {
		warn("accountTemporaryDisabled failed:", $msg->error());
	} else {
		warn "accountTemporaryDisabled succeeded";
	}
}
