# $Id: LDAP.pm 32395 2010-09-08 13:39:51Z wsl $
# $URL: https://infix.uvt.nl/its-id/trunk/sources/loandesk/lib/UvT/Loandesk/LDAP.pm $

use utf8;
use strict;
use warnings FATAL => 'all';

package UvT::Loandesk::LDAP::Config;

use Net::LDAP::Filter;

use Xyzzy -self;

field ldap_server => 'ldap://localhost';
field ldap_base => sub { die "no LDAPBase configured\n" };
field ldap_filter => undef;
field ldap_attribute => 'uid';
field ldap_username => undef;
field ldap_password => sub { die "no LDAPPassword configured\n" if shift->ldap_username };
field ldap_capath => undef;
field ldap_cafile => undef;

*set_ldapserver = *ldap_server;
*set_ldapbase = *ldap_base;
*set_ldapattribute = *ldap_attribute;
*set_ldapusername = *ldap_username;
*set_ldappassword = *ldap_password;

sub set_ldapfilter {
	my $filter = shift;

	my $compiled = eval { new Net::LDAP::Filter($filter) };
	die "error parsing LDAP filter '$filter': $@" if $@;
	die "error parsing LDAP filter '$filter'\n" unless $compiled;

	$self->ldap_filter($compiled);
}

sub set_ldapcafile {
	my $value = shift;
	die "LDAPCAfile '$value' not accessible\n"
		unless -r $value;
	die "LDAPCAfile '$value' not a file\n"
		unless -f _;
	$self->ldap_cafile($value);
}

sub set_ldapcapath {
	my $value = shift;
	die "LDAPCApath '$value' not accessible\n"
		unless -r $value;
	die "LDAPCApath '$value' not a directory\n"
		unless -d _;
	$self->ldap_capath($value);
}

package UvT::Loandesk::LDAP;

use URI;
use Net::LDAP;
use Net::LDAP::Filter;

use Clarity -self;

field cfg;
field server => sub { shift->cfg->ldap_server };
field base => sub { shift->cfg->ldap_base };
field filter => sub { shift->cfg->ldap_filter };
field attribute => sub { shift->cfg->ldap_attribute };
field username => sub { shift->cfg->ldap_username };
field password => sub { shift->cfg->ldap_password };
field cafile => sub { shift->cfg->ldap_cafile };
field capath => sub { shift->cfg->ldap_capath };

sub bind {
	my $ldap = shift // $self->connection;
	my $username = $self->username;
	return $ldap->bind($username, password => $self->password)
		if $username;
	return $ldap->bind;
}

sub connection {
	# create a, possible cached, LDAP client object
	my $ldap = $self->{connection};
	if(defined $ldap) {
		eval { $self->bind($ldap) };
		if($@) {
			my $server = $self->server;
			warn "LDAP server $server: $@"
				unless $@ =~ /^Unexpected EOF /;
			undef $ldap;
			delete $self->{connection};
		}
	}
	unless(defined $ldap) {
		my $server = $self->server;
		my %options = (onerror => 'die', timeout => 10);
		my $host = $server;
		if(index($server, '://') == -1) {
			$host =~ s/:.*//;
		} else {
			my $uri = new URI($server);
			$host = $uri->host;
		}
		if($host eq 'localhost') {
			$ldap = new Net::LDAP($server, %options)
				or die "Connecting to $server: $@";
		} else {
			my %ssl = (
					verify => 'require',
					capath => $self->capath,
					cafile => $self->cafile,
				);
			warn "neither LDAPCAfile nor LDAPCApath configured\n"
				unless $ssl{cafile} || $ssl{capath};
			$ldap = new Net::LDAP($server, %options, %ssl)
				or die "Connecting to $server: $@";
			$ldap->start_tls(%ssl)
				unless $ldap->cipher;
			die "STARTTLS failed on LDAP server $server\n"
				unless $ldap->cipher;
			die "Can't verify LDAP server name as '$host'\n"
				unless $ldap->socket->verify_hostname($host, 'ldap');
		}
		$self->bind($ldap);
		$self->{connection} = $ldap;
	}

	return $ldap;
}

sub search {
	my ($uid, $attrs) = @_;

	confess 'idiot' if ref $uid;

	return undef unless defined $uid;

	# attribute to search on
	my $attribute = $self->attribute;
	my $struct = {equalityMatch => {attributeDesc => $attribute, assertionValue => $uid}};

	# create a, possible cached, filter
	my $filter = $self->filter;
	$struct = {and => [$struct, $filter]}
		if $filter;

	my $search = bless($struct, 'Net::LDAP::Filter')->as_string;

	return $self->connection->search(
			base => $self->base,
			filter => $search,
			attrs => $attrs
		);
}

sub anr {
	my $uid = shift;

	my $res = $self->search($uid, ['employeeNumber']);

	my $num = $res->count
		or return;

	die "multiple ANRs returned for '$uid'\n"
		if $num > 1;

	return $res->entry->get_value('employeeNumber');
}
