# $Id: Setpwd.pm 37763 2012-09-06 14:41:46Z anton $
# $URL: https://svn.uvt.nl/its-id/trunk/sources/pwdmodifier/keymaster/lib/Setpwd.pm $
package Setpwd;
use strict;
use warnings FATAL=> 'all';
use Data::Dumper;
use Baseobject;
use Carp;
use Encode qw(encode);
use MIME::Base64;
use Digest::SHA1;
use FileHandle;
use Fcntl qw(:DEFAULT :flock);
use IO::Seekable;
use Lock;
use Rcrypt;
use Carp;

our @ISA=('Baseobject' );
our @EXPORT = qw (verbosity);
my $FAIL = "keymaster status: Failure:";

sub new {
  my $pkg = shift;
  my $self = bless (new Baseobject(@_),$pkg);
  #
  # NOTE NOTE NOTE allow all fields is set to  YES
  #
  $self->{allowallfields}=1;

  $self->needs(
	       [ qw (
		      ipaddress anr ldaphost sn uid nosvnfile ldapservers capath oracle_home  userinfo
		      oracle_pwd_read_table oracle_pwd_write_table allowpwdrepeat sql passwordLeader
		      ) ]
	       );

  $self->allows(
	  [ qw (challenge time oracle_testanr mailalso2 skiporacle dryrun untouchables newPwd
            silentsshclients silentclients sync-passwords reset-password contact language
		     syncOnlyIfNeeded verifyPasswordSet adminMailAddress ldapRetries ) ]
	  );

  $self->allows({'tildeusername' => ''});
  my $cf=shift;
  $self->{cf} = $cf;
  $self->fields(@_);
  $self->fields($self->getparms($cf));
  $self->checkneeds();
  return $self if ($self->error());

  $self->getaccounts();
  return $self if $self->error();

  $self->{fields}->{allowpwdrepeat} = 1 if $self->{fields}->{stealthmode};
  $self->saveLdapObjects();
  $self->checkAccountInSQL();
  return $self;
}

sub error {
# NB In tegenstelling tot de meeste Baseobjects
# sterft deze error()
	my $self = shift;

	if (@_ && $_[0]) {
		$self->{error} .= join ('',@_);
		die ("@_\n");
	}
	else  {
		my ($package,$file,$line) = caller;
		return "$package: ",$self->{error};
	}
}

sub verifyPasswordSet {
    # Verify that the new password can actually be used
	my $self = shift;
	my $host = shift;
	my $port =  636; #fixme!
	my $pwd = $self->{fields}->{newPwd};
	my $uid = $self->{fields}->{userinfo}->{uid};
	my @hosts = (keys %{$self->{ldapobjects}});
	@hosts = ($host) if $host;

	my $errorString = '';

	foreach my $host (@hosts) {
		my $ldap;
		my $retries = $self->{fields}->{ldapRetries};
		my $attempt = 0;
		my $state;
		while (!$ldap && $attempt++ < $retries) {

			$ldap=Net::LDAPS->new(
				$host,
#           FIXME: verify uitgezet voor AD
#           'verify' => 'require',
				'capath' => $self->{fields}->{capath},
				'port'   => $port,
				);
			$state = $@;
			warn "connect attempt $attempt FAILED for: $host ($uid)" unless $ldap;
		}

		unless ($ldap) {
			warn "ERROR Could not connect to $host:$port ($uid) $state";
			$errorString .= "$uid\@$host: NOCONNECT ";
			next;
		}
		$self->debug("Connection to $host OK");
		foreach my $entry ( @{$self->{ldapobjects}->{$host}->{entries}}){
			my $dn = $entry->dn;

#			next unless $self->isPersonClass($entry, $host);
			my $msg = $ldap->bind( $dn, password=>$pwd );
			if ($msg->code)	{
				warn "    verification bind FAILED for: $dn ($uid) on ldapserver: $host:$port ",$msg->error();
				$errorString .= "$uid\@$host: NOBIND ";
			}else {
				warn "    verification bind OK for dn: '$dn' ($uid) on ldapserver: $host:$port";
			}
		}
	}

	$self->error($errorString) if ($errorString);
}

sub saveLdapObjects {
  my $self=shift;

  foreach my $host ( keys %{$self->{ldapaccounts}}) {
    $self->debug ("host $host");
    unless (exists ($self->{ldapaccounts}->{$host}->{protocol})) {
      $self->verbose( "IGNORING $host");
      next ;
    }
    $self->saveldapobject($host);
  }
}


sub fetchCurrentSHAPassword {
	my $self = shift;
	my $anr = $self->{fields}->{anr};
	my $sql = $self->{fields}->{sql};
	return $sql->getCurrentSHAPassword($anr);
}

sub fetchCurrentRcryptPasswordViaSQL {
	my $self = shift;
	my $anr = $self->{fields}->{anr};
	my $sql = $self->{fields}->{sql};
	my $pwd = $sql->getCurrentRcryptPassword($anr);
	$pwd =~ s/^\{rcrypt\}// if $pwd;
	return $pwd;
}

sub fetchInitialPassword {
	my $self = shift;
	my $anr = $self->{fields}->{anr};
	my $sql = $self->{fields}->{sql};
#	my $initialPassword = $sql->getInitialPassword($anr);
#	my $Rcrypted;
#	unless ($initialPassword) {
	my 	($Rcrypted, $initialPassword) = $self->fetchCurrentLdapPassword();
#	}
	return $initialPassword;
}

sub fetchCurrentSQLPassword {
	my $self = shift;
	my $anr = $self->{fields}->{anr};
	my $rcryptPwd = $self->fetchCurrentRcryptPasswordViaSQL();
    if ($rcryptPwd) {
		return Rcrypt::decrypt($rcryptPwd, $self->{rcryptkey});
	} else {
		return $self->fetchInitialPassword();
	}
}

sub fetchCurrentLdapPassword {
	# Fetch the current password from the 'ldaphost' ldapobject
	my $self = shift;
	my $ldaphost = $self->{fields}->{ldaphost};
	my $anr = $self->{fields}->{anr};
	$ldaphost =~ s{ldaps?://}{};

	$self->debug("ldaphost: $ldaphost");
	my $ldapobject = $self->{ldapobjects}->{$ldaphost};

	my $entries =  $ldapobject->{entries};
	my $entry = @$entries[0];
	die "No entry found for ldaphost: $ldaphost" unless ($entry);

	my $rcryptpwd =	$entry->get_value('rcryptpassword');
	die "NO RCRYPTPWD FOUND for: $self->{fields}->{anr}" unless defined($rcryptpwd);
	die "userError: Account DISABLED, invalid rcrypt found" unless $rcryptpwd=~s/^\{rcrypt\}//;

	my $decrypted = Rcrypt::decrypt($rcryptpwd,$self->{rcryptkey});
	if ($decrypted) {
 		$self->debug("Current ldapPassword retreived for: $anr");
	} else {
		warn "Error: Current ldapPassword NOT retreived for: $anr";
	}
	return ($rcryptpwd, $decrypted);
}

sub synchronisePassword {
	my $self = shift;
	my $ret;

	$self->{fields}->{allowpwdrepeat} = 1;
	$self->{fields}->{skiporacle} = 1 if $self->{fields}->{passwordLeader} eq 'oracle';

	unless ( grep /^$self->{fields}->{passwordLeader}$/, qw (oracle ldap rcrypt)){
		die "$FAIL incorrect passwordleader: $self->{fields}->{passwordLeader}";
	}
	my $passwordLeader = $self->{fields}->{passwordLeader};

	$ret->{oracle} = $self->fetchCurrentSQLPassword();
	($ret->{rcrypt}, $ret->{ldap}) = $self->fetchCurrentLdapPassword();

	$self->{fields}->{newPwd} = $ret->{$passwordLeader};
	if ($self->{fields}->{syncOnlyIfNeeded}){
		if(exists ($ret->{rcrypt}) and exists ($ret->{oracle}) and exists ($ret->{ldap})){
			if(($ret->{oracle} eq $ret->{ldap}) and ($ret->{oracle} eq $ret->{rcrypt})) {
				warn "No need to sync according to baseldapserver and Oracle";
				return;
			}
		}
	}

	# If the passwordLeader did not provide a password get the current decrypted password from ldap or fail.
	unless ($self->{fields}->{newPwd}){
		if ($ret->{ldap}) {
			$self->{fields}->{newPwd} = $ret->{ldap}
		} else {
			die("Error: No password found for: $self->{fields}->{anr}");
		}
	}
	$self->setPassword();
}


sub getparms {
	my $self = shift;
	my $cf = shift;
	my $h;
	my $cfhash = $cf->hash();

	foreach my $el (keys %$cfhash) {
		$h->{$el} = $cfhash->{$el};
	}

	my $userinfo = $self->{fields}->{userinfo};
	foreach my $el (keys %$userinfo) {
		$h->{$el} = $userinfo->{$el};
	}
	$h->{requestfrom} = join ("\n", map {s/\n//; $_;} $cf->value('requestfrom'))
		if $cfhash->{requestfrom};

	$h->{'ldapservers'} = [($cf->value('ldapservers'))];
	$self->error("Config error ",$cf->error()) if ($cf->error());

	#optional untouchables, silent clients etc
	my $untouchables = [($cf->value('untouchables'))];

	$cf->clearerror() if $cf->error();

	if ($untouchables->[0])	{
		$h->{'untouchables'} = $untouchables;
		$self->debug( "untouchables: \"@{$h->{untouchables}}\"" ) ;
	}

	my $silentsshclients = [($cf->value('silentsshclients'))];
	$cf->clearerror() if $cf->error();

	if ($silentsshclients->[0]){
		$h->{silentsshclients} = $silentsshclients ;
		$self->debug( "Silentsshclients: \"@{$h->{silentsshclients}}\"");
	}

	my $silentclients = [($cf->value('silentclients'))];

	if ($silentclients->[0]) {
		$h->{silentclients} = $silentclients ;
		$self->debug("Silentclients: \"@{$h->{silentclients}}\"") ;
	}

	return $h;
}

sub getaccounts
{
	my $self=shift;
	my $secretfile ="$main::SYSCONFDIR/$self->{fields}->{nosvnfile}";;

	my $secretcf=new Readconfig (
		{ 'configfile'=>$secretfile,
		  'needs' => [qw (ldapaccounts rcryptkey rcryptkey2 oracleaccount) ],
		  'allows' => [ '.*' ],
		  'secret'=>1
		});

	if ($secretcf->error())
	{
		$self->error($secretcf->error()) ;
		return;
	}

	# rcryptkeys
	$self->{rcryptkey}=$secretcf->value('rcryptkey');
	$self->{rcryptkey2}=$secretcf->value('rcryptkey2');

	# privileged ldap accounts
	foreach my $hostspec ( $secretcf->value('ldapaccounts'))
	{
		next unless  ($hostspec);
		my ($host,$account,$pwd)=split(/\s*;\s*/,$hostspec,3);
		$self->{ldapaccounts}->{$host}->{account}=$account;
		$self->{ldapaccounts}->{$host}->{pwd}=$pwd;
	}
	my $oracle;
	($oracle->{host},$oracle->{account},$oracle->{pwd})=split(/\s*;\s*/, $secretcf->value('oracleaccount'));

	die $secretcf->error() if ($secretcf->error());

	$self->{oracleaccount} = $oracle;
	#combine this with the other configfile, represented in fields:

	# restrict possible subkeys, create hash from list
	my $subkeys;
	map {$subkeys->{$_}=1;}  qw (host attribs base testaccount onusernotfound hashtype deletetype
			       uid limit2 onserverdown ldapidentifier additionalfilter bindmethod passwordfield forcedrepeats);

	my $servers;

	#
	# use some unique delimitor
	#
	my $delim='{}';
	my $delimexpr=$delim; $delimexpr=~ s/(\W)/\\$1/g;

	my $ldapservers= join ($delim, @{$self->{fields}->{ldapservers}});
#  warn "ldapservers: $ldapservers";
	foreach my $xhost (split /\s*;\s*/ , $ldapservers)
	{
		my $h;
		foreach my $keypair (split /\s*$delimexpr\s*/ , $xhost)
		{
			next unless  ($keypair);
			my ($key,$value)=split(/\s*=\s*/, $keypair,2);
			if (exists( $subkeys->{$key}))
			{
				$h->{$key}=$value;
			}
			else
			{
				warn "Error: invalid subkey: \"$key\" of pair: \"$keypair\" in: $xhost\nin file: keymaster.cf";
			}
		}
		$servers->{$h->{host}}=$h;
		delete ($h->{host});
	}

	foreach my $xhost (keys %$servers)
	{
		my $port=389;
		my $protocol;
		my $host=$xhost;
		$host="ldap://$host" unless ($host =~ /^ldaps?/i);
		($protocol,$host)=split( /:\/\//, $host);
		($host,$port)= split ( /:/,$host)   if ($host =~ /:/);

		if (exists ($self->{ldapaccounts}->{$host}))
		{
            # warn "saving: ldapaccounts: $host";
			# merge two hashes
			map { $self->{ldapaccounts}->{$host}->{$_}=$servers->{$xhost}->{$_}}  (keys (%{$servers->{$xhost}}));
			$self->{ldapaccounts}->{$host}->{port}=$port;
			$self->{ldapaccounts}->{$host}->{protocol}=$protocol;
		}
		else {
			warn "NO LDAPACCOUNT SPECIFIED in file \"$secretfile\" for host: $host\n";
			confess "configError";
		}
	}

	foreach my $host (keys (%{$self->{ldapaccounts}}))
	{
		if ( !exists($self->{fields}->{userinfo}->{studentaccount})) {
			my $limit2 = $self->{ldapaccounts}->{$host}->{limit2};
			if (defined $limit2 && $limit2 eq 'studentaccounts') {
				delete $self->{ldapaccounts}->{$host};
				warn "skipping $host, because $self->{fields}->{userinfo}->{uid} is no studentaccount";
				next ;
			}
		}

		unless ($self->{ldapaccounts}->{$host}->{protocol}) {
			warn "Warning: host: $host not defined in keymaster.cf";
		}
	}
}

sub genPassword {
	my $self = shift;
	my $size = 15;
	my $string;
	open FH, '/dev/urandom'  or die "Couldn't open /dev/urandum!";
	while ($size) {
		read FH, $_, 1;
		next unless /[A-Za-z0-9]/;
		$string .= $_;
		$size--;
	}
	close FH;
	return $string;
}

sub makePassword {
	my $self = shift;
	my ($capital, $lower, $digit,$s);
	do {
		$s = $self->genPassword();
#		warn "genpassword: $s";
		$capital = $s =~ /([A-Z])/;
		$lower = $s =~ /([a-z])/;
		$digit = $s  =~ /\d/;
	} while (!($capital && $lower && $digit));
	return $s;
}


sub shapePassword {
	my ($self, $server, $cleartext) = @_;
	my $password;
	my $deletetype = $self->{ldapaccounts}->{$server}->{deletetype};

	# Als het nieuwe password leeg is, gooi het attribuut weg als deletetype eq 'delete',
	# anders genereer een random password.
	if ($cleartext eq '') {
		if (defined ($deletetype) && $deletetype eq 'delete') {
			return '' ;
		} else {
			$cleartext = $self->makePassword();
		}
	}

	# allow preference for sha, default is ssha
	my $hashtype = $self->{ldapaccounts}->{$server}->{hashtype};
	if (defined $hashtype){
		if ($hashtype eq 'sha') {
			$password = $self->sha($cleartext);
		} elsif ($hashtype eq 'none') {
			$password = $cleartext;
		} elsif ($hashtype eq 'MickeySoft') {
			$password = $self->activeDirectoryPwd($cleartext);
		} elsif ($hashtype eq 'ssha') {
			$password = $self->ssha($cleartext);
		} else {
			warn "undefined hashtype: '$hashtype' using default: ssha";
			$password = $self->ssha($cleartext);
		}
	} else {
		$password = $self->ssha($cleartext);
	}
	return $password;
}


sub setPassword {
	my $self = shift;
	# bewaar huidige password gegevens
	my ($curPwdRcrypted, $curPwdClearText) = $self->fetchCurrentLdapPassword();

	# fixme: dit kan vast wel met een hash
	if ( exists (($self->{fields}->{untouchables}))) {
		if ( grep ( /$self->{fields}->{anr}/, @{$self->{fields}->{untouchables}})) 	{
			warn  "$FAIL $self->{fields}->{anr} is an UNTOUCHABLE! The password remains untouched :) \n";
			exit 0;
		}
	}

	$self->{fields}->{dryrun} = 'no' unless (exists ($self->{fields}->{dryrun}));
	$self->debug( "Dryrun state: " .$self->{fields}->{dryrun});
	my $newfields;

	$newfields->{rcryptpassword} = $self->rcrypt($self->{fields}->{newPwd}, $self->{rcryptkey});
	$newfields->{datepasswdchanged} = time;

	#prepare oraclechanges, no commit yet
	unless ($self->{fields}->{dryrun} eq 'yes')	{
		$self->prepare_writeSQL($newfields);
		return if ($self->error);
	}

	my $fail = 0;
	my $changed = {};
	my $uid = $self->{fields}->{userinfo}->{uid};

	foreach my $server (keys (%{$self->{ldapobjects}}))	{
		last if $fail;

		my $errorString ;
		my $entries = $self->{ldapobjects}->{$server}->{entries};

		foreach my $entry (@$entries) {
			my $pwdSetAttemptsLeft = $self->{ldapaccounts}->{$server}->{forcedrepeats} || 0 + 1;
			my $dn = $entry->dn();
			my $objectclass = $entry->get_value('objectclass');
			my @objectclassList = $entry->get_value('objectclass');

			# retry loop
			do {
				$self->debug("pwdSetAttemptsLeft: $pwdSetAttemptsLeft $dn at $server");
				my $userpassword = $self->{ldapaccounts}->{$server}->{passwordfield} || 'userpassword';
				my $newpwd = $self->{fields}->{newPwd};
				my $shapedpassword = $self->shapePassword($server, $newpwd);

				# FORCEER ERROR
				# $shapedpassword  .= $pwdSetAttemptsLeft if ($pwdSetAttemptsLeft > 1);
				# warn "$pwdSetAttemptsLeft shapedpassword: $shapedpassword";
				$newfields->{$userpassword} = $shapedpassword;
				my $ldap = $self->{ldapobjects}->{$server}->{object};
				$self->debug(Dumper($newfields));

				foreach my $attrib (split (/\s*,\s*/, lc($self->{ldapaccounts}->{$server}->{attribs}))) {
					unless(exists ($newfields->{$attrib}))	{
						warn ("\tignoring invalid attrib for host $server: $attrib");
						next;
					}
					if ($newfields->{$attrib}) {
						$self->debug("\tchange: $attrib into $newfields->{$attrib}");
						$entry->replace($attrib => $newfields->{$attrib});
					} else {
						$self->debug("\tdelete: $attrib");
						$entry->delete($attrib);
					}
				}
				unless ($self->{fields}->{dryrun} eq 'yes') {
					my $msg = $entry->update($ldap);
					if ($msg->code) {
						$errorString = "Ldap update ERROR: $server, $dn ($uid) " . $msg->error();
						warn $errorString;
						$fail = 1;
					} else {
						if ($self->{ldapaccounts}->{$server}->{forcedrepeats}) {
							# voor onbetrouwbare servers: boelgakov
							sleep 2;
							eval { $self->verifyPasswordSet($server) };
							$errorString = $@;
							unless ($errorString) {
								$self->verbose( "   Password changed en verified on server: $server  $dn ($uid)");
								$changed->{$server} = $entries;
								$pwdSetAttemptsLeft = 0;
							} else {
								warn "bind verification error: $errorString";
							}
						} else {
							$self->verbose( "   Password changed on server: $server  $dn ($uid)");
						}
					}
				} else {
					warn "** NOT CHANGING PASSWORD ** for $dn ($uid) on server $server, because of dryrun";
				}
			} while (--$pwdSetAttemptsLeft >0 && $errorString && warn "retrying...");
			# Endof: retry loop

			if ($errorString) {
				$fail = 1;
				last;
			}
		} # endof foreach entry
	} #endof foreach server

#   COMMENT NEXT LINE!
#	$fail = 1;

	if ($fail){
		foreach my $server (keys (%$changed)) {
			warn "RESTORING server: \u$server";
			my $userpassword = $self->{ldapaccounts}->{$server}->{passwordfield} || 'userpassword';
			my $ldap=$self->{ldapobjects}->{$server}->{object};
			my $entryList = $self->{ldapobjects}->{$server}->{entries};

			foreach my $entry (@$entryList){
				my $dn = $entry->dn();
				foreach my $attrib (split (/\s*,\s*/, lc($self->{ldapaccounts}->{$server}->{attribs}))) {
					my $value;
					if($attrib eq $userpassword){
						$value = $self->shapePassword($server, $curPwdClearText);

					} else {
						$value = $entry->get_value($attrib);
						unless (defined ($value)){
							warn "Cannot restore attrib: $attrib for $self->{fields}->{anr}";
						}
					}
					$entry->replace($attrib, $value);
				}

				my $msg = $entry->update($ldap);
				if ($msg->code) {
					warn("Ldap RESTORE error! $server, $dn ", $msg->error());
				}
				else {
					warn "Ldap RESTORE succeeded for: $server, $dn ";
				}
			}
		}

		$self->finalize_write2oracleSQL(0);
		die  "password modification error, quiting";
	}
	else {
		$self->finalize_write2oracleSQL(1);
	}
}


sub saveldapobject
{
	my $self=shift;
	my $host=shift;
#   warn $self->inspect($self->{ldapaccounts});
	$self->debug("saveldapobject host: $host");
	my $port = $self->{ldapaccounts}->{$host}->{port};

	$self->debug ( "port: $port, host:$host");
	my $account  = $self->{ldapaccounts}->{$host}->{account};
	my $base     = $self->{ldapaccounts}->{$host}->{base};
	my $pwd      = $self->{ldapaccounts}->{$host}->{pwd};
	my $protocol = $self->{ldapaccounts}->{$host}->{protocol};

	my $ldap;

	my $retries = $self->{fields}->{ldapRetries};
	my $attempt = 0;
	my $state;
	while (!$ldap && $attempt++ < $retries) {
		warn "Connecting to ldapserver: $host, attempt: $attempt";
		$ldap=Net::LDAPS->new(
			$host,
#       FIXME: verify uitgezet voor AD
#		'verify' => 'require',
			'capath' => $self->{fields}->{capath},
			'port'   => $port,
			);
		$state = $@;
	}

	unless ($ldap)	{
		warn "Could not connect to $protocol://$host:$port $@";
		if ( exists ($self->{ldapaccounts}->{$host}->{onserverdown}) and
			 $self->{ldapaccounts}->{$host}->{onserverdown} eq lc ('skip'))	{
			warn "WARNING: Skipping host: $host, because of 'onserverdown is skip'";
			delete ($self->{ldapaccounts}->{$host});
			delete ($self->{ldapobjects}->{$host});
			return;
		} else {
			print "$FAIL Could not connect to ldapserver: $protocol://$host";
			exit 1;
		}
	}

	my $bindmethod = $self->{ldapaccounts}->{$host}->{bindmethod};
	my ($dn, $msg, $entry);
	if (defined ($bindmethod && $bindmethod eq 'dn')){
		$dn = $account;

	} else {
		my $accountbase=$base;
		if ($account=~ /,/)
		{
			($account,$accountbase) = split ( /\s*,\s*/ , $account,2);
		}

		# search for the special account
		$msg = $ldap->search(base=>$accountbase, filter=>$account);
		if ($msg->code)	{
			warn ("Error using filter: $account in ldapserver: $protocol://$host, accountbase: $accountbase, ",$msg->error);
			exit 1;
		}

		$entry=($msg->entries())[0];
		unless (defined ($entry)) {
			warn ("User \"$account\" unknown on ldapserver: $protocol://$host, accountbase: $accountbase");
			exit 1;
		}
		$dn=$entry->dn();
	}

	$msg=$ldap->bind( $dn, password=>$pwd );
	if ($msg->code)	{
		die "error binding $dn on ldapserver: $protocol://$host:$port ",$msg->error();
	}

	$self->debug( "successful bind to  ldap server: $host, binder:$dn, port: $port, base:$base");
	$self->{ldapobjects}->{$host}->{object}=$ldap;
	my $filter="employeenumber=$self->{fields}->{anr}";

	# FIXME: gedrag controleren
	if (exists(($self->{ldapaccounts}->{$host}->{limit2}))){
		return if
			($self->{ldapaccounts}->{$host}->{limit2} eq 'students') &&
			(! $self->{fields}->{userinfo}->{studentaccount} );
	}

	my $uid = $self->{fields}->{userinfo}->{uid};
	if ( exists $self->{ldapaccounts}->{$host}->{ldapidentifier}) {
		my $saved = $filter;
		my $identifier = $self->{ldapaccounts}->{$host}->{ldapidentifier};
		$filter = "$identifier=$uid";
		$self->verbose("ldapidentifier CHANGING FILTER from '$saved' with '$filter' for host: $host");
	}

	if ( exists ($self->{ldapaccounts}->{$host}->{testaccount})){
		warn ( "Overruling account: $filter with $self->{ldapaccounts}->{$host}->{testaccount} for host: $host");
		$filter = $self->{ldapaccounts}->{$host}->{testaccount};
	}

	$msg = $ldap->search(
		base => $base,
		filter => "(&($filter)(objectclass=person))"
		);
	if ($msg->code)
	{
		warn "error using filter: $filter in ldapserver: $host", $msg->error();
		$entry = undef;
	} else {
		$entry=($msg->entries())[0];
	}

	unless (defined ($entry))
	{
#       warn $self->inspect($self->{ldapaccounts}->{$host});
		if ( exists ($self->{ldapaccounts}->{$host}->{onusernotfound}) )
		{
			my $skip=0;
			if ($self->{ldapaccounts}->{$host}->{onusernotfound} eq 'skip')
			{
				$self->verbose( "SKIPPING server: $host, since user: \"$filter\" was not found");
				$skip=1;
			}
			elsif ($self->{ldapaccounts}->{$host}->{onusernotfound} eq 'skipunlessstudent')
			{
				unless (exists($self->{fields}->{userinfo}->{studentaccount})) {
					$skip=1;
					$self->verbose( "SKIPPING server: \U$host, since user: \"$filter\" was not found plus user is not student");
					$self->debug( $self->inspect($self->{fields}->{userinfo}));
				}
			}

			if ($skip)
			{
				delete ($self->{ldapaccounts}->{$host});
				delete ($self->{ldapobjects}->{$host});
				return;
			}
		}
		else
		{
			warn ("filter: \"$filter\" unknown on ldapserver: $host ");
			print ("$FAIL ldap_unknown_user, $filter\n");
			exit 0;
		}
	}
	else
	{
		$self->verbose( "Found user: $filter,  on: $host");

		# save the current attribs for restoring purposes for this user
		$dn=$entry->dn();
		$self->{ldapobjects}->{$host}->{dn}=$dn;
		$self->{ldapobjects}->{$host}->{entries}=[$msg->entries()];
	}
}

sub checkAccountInSQL {
	my $self = shift;
	my $anr = $self->{fields}->{anr};
	my $sql = $self->{fields}->{sql};
	my $oUid = $sql->getUsername($anr);

	unless (defined ($oUid)){
		$self->error ("userError: No valid account in mibase for anr: '$anr'");
	} else {
		$self->{fields}->{uid} = $oUid;
	}
	return $oUid;
}

sub update_passwordSQL {
  my ($self, $anr, $newfields)=@_;
  my $newPwd = $self->{fields}->{newPwd};

  my $pw_rcrypt_after = $newfields->{rcryptpassword};
  my $pw_sha_after = $self->ssha($newPwd);
  chomp ($pw_sha_after);

  my $fullldaphost = $self->{fields}->{ldaphost};
  my ($ldaphost) = $fullldaphost=~/^[^\/]+\/\/([^:]+)/;

  my $pw_rcrypt_before = $self->fetchCurrentSQLPassword() || '';
  $pw_rcrypt_before = $self->rcrypt($pw_rcrypt_before ,$self->{rcryptkey});

  my $pw_sha_before = $self->fetchCurrentSHAPassword() || $self->sha('');
  my $changed_by = $self->{fields}->{userinfo}->{changed_by};
  my $username = $self->{fields}->{uid};
  my $epoch_time = $newfields->{datepasswdchanged};
  my $sql = $self->{fields}->{sql};

  $sql->setPwd( $anr, $username, $pw_sha_before, $pw_rcrypt_before,
				  $pw_sha_after, $pw_rcrypt_after, $changed_by, $epoch_time);
}

sub prepare_writeSQL {
	my $self = shift;
	my $newfields = shift;
	if ($self->{fields}->{skiporacle})  {
		warn "SKIPPING ORACLE on request";
		$self->{skiporacle} = 1;
		return;
	}

	my $anr=$self->{fields}->{anr};
	if (exists ($self->{fields}->{oracle_testanr})){
		$anr = $self->{fields}->{oracle_testanr};
		warn "Overwriting ORACLE anr with '$anr'";
	}

	unless ($self->{fields}->{allowpwdrepeat}) {
		if ($self->usedPasswordBefore()) {
			$self->error("userError: usedpasswordbefore");
			return;
		}
	}

    $self->update_passwordSQL($anr, $newfields);
}

sub finalize_write2oracleSQL {
  #state ==1 commit, state==0 rollback
  my ($self,  $state)=@_;

  my $sql = $self->{fields}->{sql};
  my $anr = $self->{fields}->{anr};
  my $status=0;

  unless ($self->{skiporacle}) {
	  if($state) {
		  $sql->commit;
	  } else {
		  $sql->rollback;
	  }
  } else {
	  warn "Not commiting because of 'skiporacle'";
  }
}

sub usedPasswordBefore {
	my ($self) = @_;
	my $anr = $self->{fields}->{anr};
	my $sql = $self->{fields}->{sql};
	my $newpwd = $self->{fields}->{newPwd};

	# N.B. previousPasswords returns an array of refs to arrays
	my $a = $sql->previousPasswords($anr);
	foreach my $encr (@$a) {
		return 1
			if $newpwd eq Rcrypt::decrypt($encr,$self->{rcryptkey});
	}

	my $initialPassword = $sql->getInitialPassword($anr);

	# vraag niet waarom, maar het komt soms voor
	return 0 unless ($initialPassword);

	return 1
		if $newpwd eq $sql->getInitialPassword($anr);

	return 0;
}

####################################################################
sub sha {
	my ($self, $cleartext) = @_;
	my $ctx = Digest::SHA1->new;
	$ctx->add($cleartext);
	return '{SHA}' . encode_base64($ctx->digest);
}

sub ssha {
	my ($self, $cleartext) = @_;
	my $ctx = Digest::SHA1->new;
	$ctx->add($cleartext);
	my $salt = $self->genrandom(8);
	$salt = encode_base64($salt); chop($salt); chop($salt);
	$ctx->add($salt);
	return '{SSHA}' . encode_base64($ctx->digest . $salt);
}

sub activeDirectoryPwd {
	my ($self, $cleartext) = @_;

	return encode('UTF-16le', '"'.$cleartext.'"' );
}

sub genrandom
{
  my $self=shift;
  my $size=shift || 256;
  my $random;
  if (open FH, '/dev/urandom' )
  {
    read FH, $random, $size;
    close FH;
    return $random;
  }

  my $max=$size;
  my $min= 0;
  my $maxrange = 255;

  for (my $i=0;$i<$max;$i++)
  {
    $random.= int (rand ($maxrange)) + $min;
  }
  $random;
}


sub rcrypt {
  my ($self,$cleartext,$key)=@_;
  return "{rcrypt}".Rcrypt::encrypt($cleartext,$key);
}

42;

__END__
