/******************************************************************************

	worker.c -- bookkeeping for known transponder nodes
	Copyright (C) 2004  Wessel Dankers <wsl@uvt.nl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

	$Id: worker.c 7702 2005-05-25 12:13:55Z wsl $
	$URL: https://infix.uvt.nl/its-id/trunk/sources/fair/src/worker.c $

******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/socket.h>

#include "fair.h"
#include "error.h"
#include "conf.h"
#include "address.h"

#include "worker.h"

#define PRIO_MAX (~0ULL >> 1)

static worker_t worker_0 = {{0}};
static worker_t *worker_last = NULL;

static u64 worker_prio(const worker_t *wrk) {
	address_t *a;
	u64 capacity, nconn, penalty;
	avl_node_t *n;
	unless(wrk) return PRIO_MAX;
	n = wrk->prio.head;
	capacity = wrk->capacity;
	if(n && capacity) {
		a = n->item;
		unless(a) return PRIO_MAX;
		nconn = wrk->nconn + wrk->hysteresis;
		penalty = a->dead + a->disabled;
		return (nconn << 32) / capacity + (penalty << 48);
	}
	return PRIO_MAX;
}

static int worker_cmp_name(const worker_t *a, const worker_t *b) {
	assert(a && b);
	return strcasecmp(a->name, b->name);
}

static int worker_cmp_best(const worker_t *a, const worker_t *b) {
	u64 ar, br;
	assert(a && b);
	ar = worker_prio(a);
	br = worker_prio(b);
	return ar < br ? -1 : ar > br ? 1 : b->capacity - a->capacity;
}

static avl_node_t *avl_insert_node_somewhere(avl_tree_t *avltree, avl_node_t *newnode) {
	avl_node_t *node;

	assert(avltree && newnode);

	if(!avltree->top)
		return avl_insert_top(avltree, newnode);

	return avl_search_closest(avltree, newnode->item, &node) == 1
		? avl_insert_after(avltree, node, newnode)
		: avl_insert_before(avltree, node, newnode);
}

static avl_tree_t workers_name = {NULL, NULL, NULL, (avl_compare_t)worker_cmp_name, NULL};
static avl_tree_t workers_best = {NULL, NULL, NULL, (avl_compare_t)worker_cmp_best, NULL};

unsigned int worker_count(void) {
	return avl_count(&workers_name);
}

static void worker_changed(worker_t *wrk) {
	unless(wrk) return;
	avl_unlink_node(&workers_best, &wrk->best);
	avl_insert_node_somewhere(&workers_best, &wrk->best);
}

static void address_changed(address_t *addr, address_event_t evt) {
	worker_t *wrk;
	unless(addr) return;
	wrk = addr->data;
	unless(wrk) return;

	avl_unlink_node(&wrk->prio, &addr->prio);
	avl_insert_node_somewhere(&wrk->prio, &addr->prio);

	if(evt == ADDRESS_EVENT_LOAD) {
		wrk->nconn++;
		wrk->hysteresis = 0;
	} else if(evt == ADDRESS_EVENT_UNLOAD) {
		wrk->nconn--;
	}

	worker_changed(wrk);

	if(evt == ADDRESS_EVENT_LOAD && worker_last && worker_last != wrk) {
		worker_last->hysteresis = conf_Hysteresis;
		worker_changed(worker_last);
		worker_last = wrk;
	}
}

worker_t *worker_new(const char *name) {
	worker_t *wrk;
	size_t len;
	assert(name);
	len = strlen(name);
	wrk = xalloc(sizeof *wrk + len);
	*wrk = worker_0;
	strcpy(wrk->name, name);
	wrk->hysteresis = conf_Hysteresis;
	avl_init_node(&wrk->node, wrk);
	avl_init_node(&wrk->best, wrk);
	avl_init_tree(&wrk->addr, (avl_compare_t)address_cmp_addr, NULL);
	avl_init_tree(&wrk->prio, (avl_compare_t)address_cmp_prio, NULL);
	avl_insert_node(&workers_name, &wrk->node);
	avl_insert_node_somewhere(&workers_best, &wrk->best);
	syslog(LOG_INFO, "Added new worker node %s", name);
	return wrk;
}

worker_t *worker_byname(const char *name) {
	avl_node_t *n;
	worker_t *wrk;
	size_t len;
	assert(name);
	len = strlen(name);
	wrk = alloca(sizeof *wrk + len);
	*wrk = worker_0;
	strcpy(wrk->name, name);
	n = avl_search(&workers_name, wrk);
	return n ? n->item : NULL;
}

void worker_update(worker_t *wrk, int capacity, const struct sockaddr *sa, socklen_t len) {
	address_t *addr;
	address_string_t str;
	int oldcapacity;

	unless(wrk) return;

	oldcapacity = wrk->capacity;
	addr = address_byaddr(&wrk->addr, sa, len);
	if(addr) {
		wrk->capacity = capacity;
		address_update(addr);
		if(oldcapacity != capacity)
			worker_changed(wrk);
	} else if(address_authorized(sa, len)) {
		wrk->capacity = capacity;
		addr = address_new(sa, len, address_changed, wrk);
		assert(addr);
		avl_insert_node(&wrk->addr, &addr->node);
		avl_insert_node_somewhere(&wrk->prio, &addr->prio);
		address_string_sa(&str, sa, len);
		syslog(LOG_INFO, "Added new worker address %s to %s", str.host, wrk->name);
		worker_changed(wrk);
	}
}

static void worker_print(const worker_t *wrk, const address_t *best) {
	avl_node_t *n;
	address_t *addr;

	eprintf("\r%s: capacity:%d conn:%d score:%llu\033[K\n", wrk->name,
		(int)wrk->capacity, (int)wrk->nconn, (unsigned long long)worker_prio(wrk));

	for(n = wrk->addr.head; n; n = n->next) {
		addr = n->item;
		assert(addr);
		eprintf("\tconn:%d errors:%d dead:%d disabled:%d%s\n",
			(int)addr->nconn, (int)addr->errors, 
			(int)addr->dead, (int)addr->disabled,
			addr == best ? " *" : "");
	}
}

address_t *worker_bestaddress(void) {
	avl_node_t *n;
	worker_t *wrk;
	n = workers_best.head;
	if(!n)
		return NULL;
	wrk = n->item;
	unless(wrk) return NULL;
	n = wrk->prio.head;
	if(!n)
		return NULL;
	if(conf_Debug)
		for(n = workers_best.head; n; n = n->next)
			worker_print(n->item, n->item);
	return n->item;
}
