<?php
require_once 'CryptoElement.php';
require_once 'CGE_UnsupportedMethodException.php';

/**
 * SPKAC is a kind of CSR, generated by the browsers' &lt;keygen&gt; tag. It is
 * specified as a part of HTML5.
 * @see http://dev.w3.org/html5/spec/Overview.html#signedpublickeyandchallenge
 *
 * Unfortunately, other than for PKCS#10 CSRs, there are no good tools to
 * parse or process SPKAC CSRs at the time of writing this code.
 * This is why most of the overriding functions in this class either throw
 * CGE_UnsupportedMethodExceptions or return default values.
 * @author tzangerl
 *
 */
class CSR_SPKAC extends CSR
{
	private static $KEYTYPE_RSA = "RSA";
	private $spkac_der;
	private $spkac_pem;

	function __construct($content)
	{
		parent::__construct($content);
		$this->spkac_der = $content;
	}

	/*
	 * --------------------------------------------------------------- *
	 *
	 *		Start of requried functions from CryptoElement.
	 *
	 * --------------------------------------------------------------- *
	 */

	/**
	 * @see CryptoElement::getLength()
	 * @throws CGE_UnsupportedMethodException
	 */
	public function getLength()
	{
		$msg = "Getting details of SPKAC-CSRs currently unsupported!";
		throw new CGE_UnsupportedMethodException($msg);
	} /* end getLength */

	/**
	 * SPKAC CSRs will only be accepted if they are submitted from browsers.
	 * In browsers we can control that the keytype is RSA and the keysize in
	 * all browsers will at least be 1024 bit.
	 *
	 * At the same time, we don't have any tools in PHP (or openssl) to
	 * properly parse SPKAC CSRs, that's why that method will always return
	 * true.
	 * @see lib/ca/CryptoElement::isValid()
	 * @return true
	 */
	public function isValid()
	{
		return true;
	} /* end isValid */

	/**
	 * Hash the whole SPKAC string as it was received by the browser. SPKAC
	 * CSRs don't include a subject, only a public key, possibly a challenge,
	 * signed by the private key that was generated by keygen.
	 *
	 * Also, we will always receive PKCS#10 CSRs and SPKAC-CSRs via different
	 * channels.
	 * @see lib/ca/CryptoElement::getPubKeyHash()
	 * @return SHA1-hash of the SPKAC object
	 */
	public function getPubKeyHash()
	{
		return sha1($this->spkac_der);
	}

	/**
	 * HTML5 keygen only supports RSA
	 * @see lib/ca/CryptoElement::getType()
	 * @return RSA
	 */
	public function getType()
	{
		return SPKAC::$KEYTYPE_RSA;
	}

	/**
	 * @see lib/ca/CryptoElement::getSubject()
	 */
	public function getSubject()
	{
		$msg = "SPKAC-CSRs do not contain subject information";
		throw new CGE_UnsupportedMethodException($msg);
	}

	/**
	 * @see lib/ca/CryptoElement::getDERContent()
	 */
	public function getDERContent($raw = true)
	{
		return $this->spkac_der;
	}

	/**
	 * @see lib/ca/CryptoElement::getPEMContent()
	 */
	public function getPEMContent($raw = true)
	{
		if (empty($spkac_pem)) {
			$this->spkac_pem = $this->der2pem($this->spkac_der);
		}

		return $this->spkac_pem;
	}

	/**
	 * Hash the whole SPKAC string to get an auth-token
	 * @see lib/ca/CSR::getAuthToken()
	 */
	public function getAuthToken()
	{
		return sha1($this->spkac_der);
	}

	public function getCSRType()
	{
		return "spkac";
	}
}
?>