use utf8;
use strict;
use warnings FATAL => 'all';

package Xyzzy::MIME::File;

use base 'IO::File';

sub close {
	my $self = shift;
	$self->seek(0, 0);
	return $self->flush;
}

package Xyzzy::MIME::TempFile;

use base 'Xyzzy::MIME::File';

sub new {
	my $proto = shift;
	my $self = $proto->SUPER::new
		or return undef;
	open($self, '+<', undef)
		or return undef;
	return $self;
}

sub as_string {
	my $self = shift;
	$self->seek(0, 0);
	local $\;
	my $str = $self->getline // ($self->eof ? '' : die $!);
	$self->seek(0, 0);
	return $str;
}

sub size {
	return -s shift;
}

sub reader {
	open(my $fh = new IO::File, '<&', shift)
		or return undef;
	return $fh;
}

package Xyzzy::MIME::CoreFile;

use base 'Xyzzy::MIME::File';

our $core_max = 4096;

# uses either a scalar or a temporary file as backend, depending on the size

sub new {
	my $proto = shift;
	my $str = '';
	my $self = $proto->SUPER::new(\$str, '+<')
		or return undef;
	${*$self}{str} = \$str;
	return $self;
}

sub as_string {
	my $self = shift;
	return ${${*$self}{str}};
}

sub size {
	my $self = shift;
	return length ${${*$self}{str}};
}

sub _convert {
	my $self = shift;
	my $ref = ${*$self}{str};
	return if length($$ref) < $core_max;
	open($self, '+>', undef)
		or die "Can't open a temporary file: $!\n";
	$self->SUPER::print($$ref)
		or die "Can't write to temporary file: $!\n";
	delete ${*$self}{str};
	bless $self, 'Xyzzy::MIME::TempFile';
}

sub say {
	my $self = shift;
	_convert($self);
	return $self->SUPER::say(@_)
}

sub print {
	my $self = shift;
	_convert($self);
	return $self->SUPER::print(@_)
}

sub printf {
	my $self = shift;
	_convert($self);
	return $self->SUPER::printf(@_)
}

sub format_write {
	my $self = shift;
	_convert($self);
	return $self->SUPER::format_write(@_)
}

sub write {
	my $self = shift;
	_convert($self);
	return $self->SUPER::write(@_)
}

sub syswrite {
	my $self = shift;
	_convert($self);
	return $self->SUPER::syswrite(@_)
}

sub close {
	my $self = shift;
	_convert($self);
	return $self->SUPER::close(@_)
}

sub reader {
	my $self = shift;
	return new IO::File(${*$self}{str}, '<');
}

package Xyzzy::MIME::Body;

# a body that uses Xyzzy::MIME::File as a backend

use base 'MIME::Body';

sub init {
	my $self = shift;
	$self->{XMB_class} = shift // 'Xyzzy::MIME::CoreFile';
}

sub open {
	my ($self, $mode) = @_;
	if($mode eq 'w') {
		my $class = $self->{XMB_class};
		return $self->{XMB_fh} = $class->new;
	} else {
		my $fh = $self->{XMB_fh}
			or return new IO::File(\'', '<');
		return $fh->reader;
	}
}

sub size {
	my $fh = shift->{XMB_fh}
		or return undef;
	return $fh->size;
}

sub as_string {
	my $fh = shift->{XMB_fh}
		or return undef;
	return $fh->as_string;
}

sub path {
	return undef;
}

package Xyzzy::MIME::Parser;

use base 'MIME::Parser';

sub new_body_for {
	my (undef, $head) = @_;

	return new Xyzzy::MIME::Body('Xyzzy::MIME::TempFile')
		if defined $head->mime_attr('content-disposition.filename');

	return new Xyzzy::MIME::Body;
}

1;
